
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 25 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_AESFLOW.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/


#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_UTIL_OTF_Funcs.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_FLOW_Defines.h"
#include "gen.h"
#include "CRYS_INIT_TST.h"
#include "CRYS_FLOW.h"
#include "CRYS_FLOW_error.h"
#include "DX_VOS_Mem.h"
#include "FLOW_error.h"

#if TST_USE_FLOW_DATA


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern DxUint8_t 	         TST_Input_Data[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];

extern DxUint8_t 	         TST_Input_Data_ForCyclicOperation[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];
/* input data */
DxUint64_t	                 TST_FLOW_InputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* output data */
DxUint64_t	                 TST_FLOW_OutputData[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

DxUint64_t	                 TST_FLOW_OutputDataForIntegratedEngines[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* input data size */
DxUint32_t                   TST_FLOW_InputDataSize = TST_FLOW_DATA_VEC_SIZE_IN_BYTS; 

/* flow params info */
CRYS_FLOW_Params_t           TST_FLOWParamsInfo;

/* AES engine data */
CRYS_FLOW_AESEngineData_t    TST_FLOWAESEngineData;

/* common data for one test vector */
TST_FLOW_TestCommonDataStruct TST_TestData;

/* output data of HASH/AES_MAC result */
CRYS_AES_MAX_MAC_RESULT_t     TST_FLOW_AESMAC_OutputData;

/* flow input buffers */
CRYS_FLOW_BufferData_t        TST_InputBuffers[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];

/* flow output buffers */
CRYS_FLOW_BufferData_t        TST_OutputBuffers[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];


// indication if Flow_Finish operation should be executed with finish size = 0:
DxBool_t                      TST_FLOW_IsFinishSizeZero = DX_FALSE;

/************* Private function prototype ***********************/

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InitParams
 *  
 * Inputs:
 *  TST_index           - test index
 *  isNotAddBufferFlow  - indication if flow is with added buffers or not
 *  pAcceptanceTestData - input test data
 *  pFlowInfo           - output flow info, intialized from test data
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function initializes parameters for FLOW operation
 * Algorithm:     
 *	Initialize pFlowInfo struct with values from the test vector.
 ******************************************************************************/                                
void CRYS_ATP_TST_FLOW_InitParams ( int                            TST_index, 
								    DxBool_t                       isNotAddBufferFlow,
									TST_FLOW_TestCommonDataStruct *pAcceptanceTestData,
									CRYS_FLOW_Params_t            *pFlowInfo,
									CRYS_FLOW_Engine_t             flowEngineType)
									
{
    /********* TST Variables Definitions ************/
    DxUint16_t i = 0;
    DxUint16_t outputBlockSize = 0;
    
    /* Initialize flow info parameter with input and output buffers and other parameters */
    /* from given test data, which is contained in pAcceptanceTestData parameter         */
    /*************************************************************************************/
    
	/* input block size */
	DxUint16_t blockSize = TST_FLOW_InputDataSize/pAcceptanceTestData->inputBuffersArraySize;
	
	if (pAcceptanceTestData->outputBuffersArraySize != 0)
	    outputBlockSize = TST_FLOW_InputDataSize/pAcceptanceTestData->outputBuffersArraySize;
	else
	    outputBlockSize = 0;
    
    /* initialize input and output interrupts id */
    pFlowInfo->inputInterruptId   = pAcceptanceTestData->inputInterruptId;
    
    pFlowInfo->outputInterruptId  = pAcceptanceTestData->outputInterruptId;        
        
    /* if the data in the buffers are in words and not bytes */ 
    pFlowInfo->wordsBuffersOrderFlag = pAcceptanceTestData->wordsBuffersOrderFlag;        

    /* array of pointers to the input buffers */
    if (isNotAddBufferFlow == DX_TRUE)
    {
        /* in static flow initiaize with number of all input buffers */
        pFlowInfo->inputBuffersArraySize = pAcceptanceTestData->inputBuffersArraySize;
    }
    else
    {
        /* in flow with added buffers initialize with one input buffer at the beginning */
        pFlowInfo->inputBuffersArraySize = 1;
    }
    
    pFlowInfo->inputBuffersArrays = TST_InputBuffers;
    
    /* initialize flow input buffers */
    for (i = 0; i < pFlowInfo->inputBuffersArraySize; ++i)
    {
        TST_InputBuffers[i].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + i*blockSize;
        
        TST_InputBuffers[i].bufferSize = blockSize;                
        
        /* in flow with added buffers initialize only with one input buffer  */
        if (isNotAddBufferFlow == DX_FALSE)
            break;
    }
    
  
    /* flag if  endian order of the input buffers is big */
    pFlowInfo->inputBigEndianOrderFlag = pAcceptanceTestData->inputBigEndianOrderFlag;
    
     
    /* array of pointers to the output buffers */
    if (pAcceptanceTestData->outputBuffersArraySize != 0)
    {
        /* if flow is with added buffers initialize output buffers array size to be 1: */
        if (isNotAddBufferFlow == DX_TRUE)
            pFlowInfo->outputBuffersArraySize = pAcceptanceTestData->outputBuffersArraySize;
        else
            pFlowInfo->outputBuffersArraySize = 1;
        
        
        pFlowInfo->outputBuffersArrays = TST_OutputBuffers[flowEngineType];
    
        /* initialize flow output buffers */
        for (i = 0; i < pFlowInfo->outputBuffersArraySize; ++i)
        {
            TST_OutputBuffers[flowEngineType][i].buffer_ptr = (DxUint8_t*)TST_FLOW_OutputData[flowEngineType] + i*outputBlockSize;
            
            TST_OutputBuffers[flowEngineType][i].bufferSize = outputBlockSize;                
            
            /* in flow with added buffers initialize only with one output buffer  */
            if (isNotAddBufferFlow == DX_FALSE)
                break;
        }
    }
    else
    {
       pFlowInfo->outputBuffersArraySize = 0;
        
    }
    
    /* flag if  endian order of the output buffers is big */
    pFlowInfo->outputBigEndianOrderFlag = pAcceptanceTestData->outputBigEndianOrderFlag;
 }
 
 /*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InterruptUp
 *  
 * Inputs:
 *  TST_InterruptId       - interrupt id
 
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function araise the given interrupt
 * Algorithm:  Write to the corresponding register   
 ******************************************************************************/                                
 void CRYS_ATP_TST_FLOW_SendInterrupt ( int  TST_InterruptId )
 {
    #ifdef DX_ARM1176
    	DX_GEN_WriteRegister(0x80008018,(1 << TST_InterruptId));
    #else
    	DX_GEN_WriteRegister(0xC0008018,(1 << TST_InterruptId));
    #endif		
 }

 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_GetKeySizeInBytes
 *  
 * Inputs:
 *  keySize - key size enumeration
 *  None
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function returns key size in bytes
 * Algorithm:     
 ******************************************************************************/                                
DxUint32_t CRYS_ATP_TST_FLOW_GetKeySizeInBytes (CRYS_AES_KeySize_t keySize)
{
    /* Get key size in bytes */
 	/*****************************/
 	
    switch (keySize)
    {
        case CRYS_AES_Key128BitSize:
            return TST_AES_Key128Size_InBytes;
            
            
        case CRYS_AES_Key192BitSize:
            return TST_AES_Key192Size_InBytes;
            
                
        case CRYS_AES_Key256BitSize:
            return TST_AES_Key256Size_InBytes;
            
            
        case CRYS_AES_Key512BitSize:
            return TST_AES_Key512Size_InBytes;
           
            
        default:
            return TST_AES_Key512Size_InBytes;                        
           

    }
}


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InitAESEngineData
 *  
 * Inputs:
 *  TST_index - test index
 *  None
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function initializes engine data for AES FLOW operation
 * Algorithm:     
 *	This function initializes AES flow engine data struct with values from the test 
 *  vector.
 ******************************************************************************/                                
void CRYS_ATP_TST_FLOW_InitAESEngineData ( int TST_index ) 
{
    /********* TST Variables Definitions ************/
    DxUint32_t keySizeInBytes = 0;
    
    /* Initializing AES engine data */
 	/********************************/
    
    /* initialize IV counter */
    DX_VOS_FastMemCpy(TST_FLOWAESEngineData.IVCounter_ptr, 
                      TST_AESFLOW_Acceptance_TestData[TST_index].TST_IvCounter,
                      CRYS_AES_IV_COUNTER_SIZE_IN_BYTES);
           
           
    /* convert key size enumeration to key size in bytes */ 
    keySizeInBytes = CRYS_ATP_TST_FLOW_GetKeySizeInBytes(TST_AESFLOW_Acceptance_TestData[TST_index].TST_KeySize);
    
    /* initialize key value */
    DX_VOS_FastMemCpy(TST_FLOWAESEngineData.Key_ptr, 
                      TST_AESFLOW_Acceptance_TestData[TST_index].TST_Key,
                      keySizeInBytes);
  
    TST_FLOWAESEngineData.KeySize       = TST_AESFLOW_Acceptance_TestData[TST_index].TST_KeySize;
    TST_FLOWAESEngineData.OperationMode = TST_AESFLOW_Acceptance_TestData[TST_index].TST_OperationMode;
     
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_InitAESFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  flowType           - flow type
 *  TST_MasterTestName - test name
 *  TST_AESFlowId      - output AES flow id
 *  pAESFlowParamsInfo - flow params info
 *  encryptMode        - encrypt/decrypt mode
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function starts AES flow.
 *
 * Algorithm:     
 *	1. Initialize Flow info parameter.
 *  2. Init AES flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_InitAESFlow ( int                     TST_index,
                                       CRYS_FLOW_Type_t        flowType,
                                       char                   *TST_MasterTestName, 
                                       DxUint32_t             *TST_AESFlowId,
                                       CRYS_FLOW_Params_t     *pAESFlowParamsInfo,
                                       CRYS_AES_EncryptMode_t  encryptMode) 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    TST_TestData.inputBigEndianOrderFlag  = TST_AESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_AESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_AESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_AESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_AESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;
	
    /* Initializing Flow info parameter */
 	/************************************/
    
    /* initialize flow init parameters */
 	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, pAESFlowParamsInfo, CRYS_FLOW_AES);
 	    
 	/* initialize AES engine data */
 	CRYS_ATP_TST_FLOW_InitAESEngineData(TST_index);
         	
    TST_FLOWAESEngineData.EncryptDecryptFlag = encryptMode;
    
    if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
     	(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
     	(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
     	
 	{
 	    pAESFlowParamsInfo->outputBuffersArrays    = DX_NULL;
 	    pAESFlowParamsInfo->outputBuffersArraySize = 0;
 	}
    
    /* Init AES flow */
 	/*****************/
 	TST_ATP_Error = CRYS_FLOW_AESInit(flowType,
                 	                  pAESFlowParamsInfo,
                                      &TST_FLOWAESEngineData,  
                                      TST_AESFlowId);
                      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_InitAES is done\n"));
                        
    if (TST_FLOWAESEngineData.EncryptDecryptFlag == CRYS_AES_Encrypt)
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AESInit encrypt",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
    else
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AESInit decrypt",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    return TST_TestStatus;
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FinishAESFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_AES            - flow id
 *  TST_MasterTestName - test name
 *  blockSize          - block size for finish
 *  pAESFlowParamsInfo - flow params info
 *  encryptMode        - encrypt/decrypt mode
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function finishes AES flow and check results.
 *
 * Algorithm:     
 *	1. Finish Flow
 *  2. Get results
 *  3. Compare flow output with expected output
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FinishAESFlow ( int                    TST_index, 
                                         DxUint32_t             TST_AESFlowId, 
                                         char                  *TST_MasterTestName,
                                         DxUint32_t              blockSize,
                                         CRYS_FLOW_Params_t     *pAESFlowParamsInfo,
                                         CRYS_AES_EncryptMode_t  encryptMode ) 
{
    /********* TST Variables Definitions ************/
    DxUint8_t    *pExpectedOut  = DX_NULL;
    CRYSError_t  TST_TestStatus = TESTPASS;
    DxUint32_t   j = 0; 
    
    /* Finish AES flow with size of the last input buffer */
    /******************************************************/
    TST_ATP_Error = CRYS_FLOW_Finish(TST_AESFlowId,
                                     blockSize);  
                     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
                     
    /* Get results */
    /***************/
    do
    {

        TST_ATP_Error  = CRYS_FLOW_GetResults(TST_AESFlowId,          
                                              TST_FLOW_AESMAC_OutputData,
                                              sizeof(CRYS_AES_MAX_MAC_RESULT_t));
                                                                                                                      

    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);

							          
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  

    
    if (encryptMode == CRYS_AES_Encrypt)
		pExpectedOut = TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
	else
		pExpectedOut = (DxUint8_t*)TST_Input_Data;
	
	/* Compare flow output with expected output */
	/********************************************/
    if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
		(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
		(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
	{
        /* compare and prints results of test */
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
			(DxUint8_t*)TST_FLOW_AESMAC_OutputData,
			pExpectedOut,
			CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);

		/*Print the outputted and then the expected result after mem compare */
		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
			"\nData outputted by engine [output buffer]-- \n",
			(DxUint8_t*)TST_FLOW_AESMAC_OutputData, 
			CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4); 

		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
			"\nData Expected by engine --\n",
			(DxUint8_t*)pExpectedOut,
			CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);                 	           
	}
	else
	{
		for (j = 0; j < pAESFlowParamsInfo->outputBuffersArraySize; ++j)
		{
			/* compare and prints results of test */
    		TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
													pAESFlowParamsInfo->outputBuffersArrays[j].buffer_ptr,
                                     				pExpectedOut + 	j*pAESFlowParamsInfo->outputBuffersArrays[j].bufferSize,
                                     				pAESFlowParamsInfo->outputBuffersArrays[j].bufferSize);
	                                    
	                                     			
			/*Print the outputted and then the expected result after mem compare */
			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
									   "\nData outputted by engine [output buffer]-- \n",
									   (DxUint8_t*)pAESFlowParamsInfo->outputBuffersArrays[j].buffer_ptr, 
									   pAESFlowParamsInfo->outputBuffersArrays[j].bufferSize); 

	 		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
									  "\nData Expected by engine --\n",
									   (DxUint8_t*)pExpectedOut + 	j*pAESFlowParamsInfo->outputBuffersArrays[j].bufferSize,
									   pAESFlowParamsInfo->outputBuffersArrays[j].bufferSize);                 	                                     			
		}
	}
			
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    return TST_TestStatus;
}

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_CheckAESOperation
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function checks some AES operation
* Algorithm:     
* 1. Execute some AES operation 
* 2. Compare result to expected one
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_CheckAESOperation () 
{
	/********* TST Variables Definitions ************/
	DxUint32_t   ExpectedSize = TST_FLOW_InputDataSize;
	
	/* use test index = 0 */
	DxUint32_t   TST_index = 0;
	CRYSError_t  TST_TestStatus = TESTPASS;


	/* Run CRYS_AES function */
	/*************************/
	CRYS_AES( TST_AESFLOW_Acceptance_TestData[0].TST_IvCounter,
		TST_AESFLOW_Acceptance_TestData[0].TST_Key,
		TST_AESFLOW_Acceptance_TestData[0].TST_KeySize,
		CRYS_AES_Encrypt,
		TST_AESFLOW_Acceptance_TestData[0].TST_OperationMode,
		TST_Input_Data,
		TST_FLOW_InputDataSize,
		(DxUint8_t*)TST_FLOW_OutputDataForIntegratedEngines);
		
		
    if ((TST_AESFLOW_Acceptance_TestData[0].TST_OperationMode == CRYS_AES_MAC_mode)      ||
     	(TST_AESFLOW_Acceptance_TestData[0].TST_OperationMode == CRYS_AES_XCBC_MAC_mode) ||
     	(TST_AESFLOW_Acceptance_TestData[0].TST_OperationMode == CRYS_AES_CMAC_mode))	
    {
        ExpectedSize = CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4;
    }

	/* compare AES result with one expected */
	/****************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		TST_FLOW_OutputDataForIntegratedEngines,
		TST_AESFLOW_Acceptance_TestData[0].TST_Expected_output_data,
		ExpectedSize);					      

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CheckAESOperation is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CheckAESOperation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
	return TST_ATP_Error;

}


/************************ Public Functions ******************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name
 *  flowType           - flow type (static or dynamic)
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES FLOW API
 * Algorithm:       
 *  1. Initialize AES flow with type and other parameters
 *  2. Send interrupts that input and output buffers are ready
 *  3. Finish AES flow, read data from the output buffers and compare to expected results
 *  4. Terminate flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance ( int              TST_index, 
                                                       char            *TST_MasterTestName,
                                                       CRYS_FLOW_Type_t flowType) 
{

    /********* TST Variables Definitions ************/
    CRYSError_t             	TST_TestStatus = TESTPASS;
    
 	/* flow id */
    DxUint32_t                   flowId = 0;
    int                          i = 0;
    DxUint16_t                   blockSize;
	

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                                                                              TST_OperationMode = 0 == Encrypt
                                                                                                              TST_OperationMode = 1 == Decrypt*/
    {
     	/* input block size */
        blockSize = TST_FLOW_InputDataSize/TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		
     	/* initialize input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
		{
			/* in modes below there is only encrypt option */
			if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CTR_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
				break;

			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
		}
		
		/* Initialize AES flow parameters and init AES flow by calling function */
		/* CRYS_FLOW_AESInit                                                    */
		/************************************************************************/
		TST_TestStatus = CRYS_ATP_TST_InitAESFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo, (CRYS_AES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
     	
     	#if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Send interrupt that input/output buffers are ready */
        /******************************************************/
        
        /* input interrupts: */
        if (TST_FLOW_IsFinishSizeZero == DX_TRUE)
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
	            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
	          
            	          
            blockSize = 0;	            
        }
        else
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 1; ++i)
	            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
        }
        
        
        /* Send interrupt that output buffers are ready */
		if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
		{
		    for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize; ++i)
		        CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);
		        
        }
        
 		  
        /* Finish AES flow, get results and compare results with expected  */
        /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults  */
        /*******************************************************************/
        TST_TestStatus = CRYS_ATP_TST_FinishAESFlow(TST_index, flowId, TST_MasterTestName, blockSize, &TST_FLOWParamsInfo, (CRYS_AES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
        
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
	    	return TESTFAIL;
	    #endif  
	    
	    /* Terminate flow */
	    /******************/
	    TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
    
	} /* End of TST_ATP_EncryptDecryptFlag*/ 
	                            
    return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static AES FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute AES flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     	TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance"; 
	 

	/* initialize input and output buffers */
    TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
    TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
    
  
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    /* go through all tests vectors and execute AES flow acceptance test for each vector */
    /*************************************************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	TST_TestStatus  = CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance (TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
     	
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
    	    return TESTFAIL;
    	#endif  
    }/* End of TST_index */     
    
  	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for dynamic AES FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute AES dynamic flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     	TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance"; 
     
	/* initialize input and output buffers */
    TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
    TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
    
  
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    
  	/* go through all tests vectors and execute AES flow acceptance test for each vector */
    /*************************************************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	TST_TestStatus  = CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance (TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
     	
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
    	    return TESTFAIL;
    	#endif  
    }/* End of TST_index */     
    
  	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance*/ 



/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for dynamic AES FLOW API's
 
 * Algorithm:     
 *	1. Initialize AES flow with dynamic type and other parameters
 *  2. Init AES flow
 *  3. Send interrupts that input and output buffers are ready
 *  4. Add buffers to dynamic flow
 *  5. Send interrupts that added input and output buffers are ready
 *  6. Finish AES flow
 *  7. Get results
 *  8. Terminate flow
 *  8. Read data from the output buffers and compare to expected results
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     	TST_index;
    CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance"; 
     DxUint8_t                  *pExpectedOut;
 	/* flow id */
    DxUint32_t                  flowId = 0;
    int                         j = 0; 
    DxUint32_t                  inputBuffersArraySize = 1;
    DxUint32_t                  outputBuffersArraySize = 1;
    CRYS_FLOW_BufferData_t      inputBuffersArray_ptr[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];
    CRYS_FLOW_BufferData_t      outputBuffersArray_ptr[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];
    DxUint16_t                  blockSize;
	DxUint16_t                  outputBlockSize;
    
    /* initialize input and output buffers */
    TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
    TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));
    
    
  	/************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	
		TST_TestData.inputBigEndianOrderFlag  = TST_AESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
		TST_TestData.inputBuffersArraySize    = TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		TST_TestData.inputInterruptId         = TST_AESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
		TST_TestData.outputBigEndianOrderFlag = TST_AESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
		TST_TestData.outputBuffersArraySize   = TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
		TST_TestData.outputInterruptId        = TST_AESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
		TST_TestData.wordsBuffersOrderFlag    = TST_AESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

		for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                                                                                  TST_OperationMode = 0 == Encrypt
                                                                                                                  TST_OperationMode = 1 == Decrypt*/
        {
         	/* Initialize AES flow with dynamic type and other parameters */
         	/**************************************************************/
            blockSize       = TST_FLOW_InputDataSize/TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
			
			if (TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize != 0)
			    outputBlockSize = TST_FLOW_InputDataSize/TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
			else
			    outputBlockSize = 0;
         	
         	/* prepare input buffer: */
			if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
				DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
			else
			{
				/* in modes below there is only encrypt option */
				if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
					(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CTR_mode)      ||
					(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
					(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
					break;

				DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
			}
         	
         	/* initialize flow init parameters */
         	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_FALSE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_AES);
         	
         	/* initialize AES engine data */
         	CRYS_ATP_TST_FLOW_InitAESEngineData(TST_index);
         	
         	if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
         	    (TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
         	    (TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
     	
         	{
         	    TST_FLOWParamsInfo.outputBuffersArrays    = DX_NULL;
         	    TST_FLOWParamsInfo.outputBuffersArraySize = 0;
         	}
         	
         	TST_FLOWAESEngineData.EncryptDecryptFlag = (CRYS_AES_EncryptMode_t)TST_ATP_EncryptDecryptFlag;
         	
    	    /* Init AES flow */
         	/*****************/
         	TST_ATP_Error = CRYS_FLOW_AESInit(CRYS_FLOW_TypeDynamic,
                         	                  &TST_FLOWParamsInfo,
                                              &TST_FLOWAESEngineData,  
                                              &flowId);
					
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_AESInit is done\n"));
                                
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AESInit",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
     	
            #if CRYS_ATP_TST_EXIT_ON_ERROR
                if (TST_TestStatus != TESTPASS) 
                    return TESTFAIL;
            #endif  
            
            /* Send interrupts that input and output buffers are ready */
            /***********************************************************/
            
            /* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
            
            
            /* Send interrupt that output buffers are ready */
            if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	        (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	        (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
            
			inputBuffersArraySize = TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
			outputBuffersArraySize = TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;

			/* prepare buffers to add */
			for (j = 0; j < inputBuffersArraySize - 1; ++j )
			{
				/* point to the second block of the input buffer */
				inputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j+1)*blockSize;
	                
				inputBuffersArray_ptr[j].bufferSize = blockSize;                
			}
            
           
            if (outputBlockSize != 0)
            {
                for (j = 0; j < outputBuffersArraySize - 1; ++j )
    			{
    				/* point to the second block of the output data */
    				outputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_OutputData[CRYS_FLOW_AES] + (j+1)*outputBlockSize;
    	        
    				outputBuffersArray_ptr[j].bufferSize = outputBlockSize;                
    			}
            }
            

			
            /* Add buffers to dynamic flow */
            /*******************************/
            if (outputBlockSize != 0)
            {
                TST_ATP_Error = CRYS_FLOW_AddBuffers(flowId,
                                                     inputBuffersArray_ptr,
                                                     inputBuffersArraySize - 1,
                                                     outputBuffersArray_ptr,
                                                     outputBuffersArraySize - 1);
            }
            else
            {
                TST_ATP_Error = CRYS_FLOW_AddBuffers(flowId,
                                                     inputBuffersArray_ptr,
                                                     inputBuffersArraySize - 1,
                                                     DX_NULL,
                                                     0);
            }
                                 
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_AddBuffers is done\n"));
                                
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AddBuffers",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
     	
            #if CRYS_ATP_TST_EXIT_ON_ERROR
                if (TST_TestStatus != TESTPASS) 
                    return TESTFAIL;
            #endif                                   
                                 
            /* Send interrupts that added input and output buffers are ready */
            /*****************************************************************/
            
            /* Send interrupt that input buffers are ready */
            for (j = 0; j < inputBuffersArraySize - 1; ++j )
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
                    
            blockSize = 0;    
           
           
            /* Send interrupt that output buffers are ready */
            if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	        (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	        (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
            {
                for (j = 0; j < outputBuffersArraySize - 1; ++j )
                    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);                                 
            }
            
            /* Finish AES flow */
            /******************************************************/
            TST_ATP_Error = CRYS_FLOW_Finish(flowId,
                                             blockSize);
                                             
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));
                                
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
     	
            #if CRYS_ATP_TST_EXIT_ON_ERROR
                if (TST_TestStatus != TESTPASS) 
                    return TESTFAIL;
            #endif                                                                                 

		                                           
            /* Get results */
            /***************/
            do
            {

                TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
										         (DxUint8_t*)TST_FLOW_AESMAC_OutputData,
										         CRYS_AES_BLOCK_SIZE_IN_WORDS*4);
                                                                                                                              

            }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
            
            
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
            
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
     	
            #if CRYS_ATP_TST_EXIT_ON_ERROR
                if (TST_TestStatus != TESTPASS) 
                    return TESTFAIL;
            #endif                                                                                 
            
            /* terminate flow */
            /******************/
            TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
            
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));
                                
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
     	
            #if CRYS_ATP_TST_EXIT_ON_ERROR
                if (TST_TestStatus != TESTPASS) 
                    return TESTFAIL;
            #endif                                                                                 
            
            
            /* Read data from the output buffers and compare to expected result */
            /********************************************************************/
                             
            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
				pExpectedOut = TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
			else
				pExpectedOut = (DxUint8_t*)TST_Input_Data;

			if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
			{
				/* compare and prints results of test */
				TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
					(DxUint8_t*)TST_FLOW_AESMAC_OutputData,
					pExpectedOut,
					CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);

				/*Print the outputted and then the expected result after mem compare */
				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData outputted by engine [output buffer]-- \n",
					(DxUint8_t*)TST_FLOW_AESMAC_OutputData, 
					CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4); 

				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData Expected by engine --\n",
					(DxUint8_t*)pExpectedOut,
					CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4); 
					
			    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
			
    			#if CRYS_ATP_TST_EXIT_ON_ERROR
    			if (TST_TestStatus != TESTPASS) 
    	    		return TESTFAIL;
    			#endif                  	           
			}
			else
			{
				for (j = 0; j < 1; ++j)
				{
					/* compare and prints results of test */
    	    		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
    														TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr,
    	                                     				pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
    	                                     				TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);
					                           
	    	                                     			
    				/*Print the outputted and then the expected result after mem compare */
    				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
    										   "\nData outputted by engine [output buffer]-- \n",
    										   (DxUint8_t*)TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr, 
    										   TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize); 

    		 		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
    										  "\nData Expected by engine --\n",
    										   (DxUint8_t*)pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
    										   TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);                 	                                     			

				}
			
				/* compare and prints results of test */
				for (j = 0; j < inputBuffersArraySize - 1; ++j )
				{
					TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
                            						outputBuffersArray_ptr[j].buffer_ptr,
                            						pExpectedOut + 	(j+1)*outputBuffersArray_ptr[j].bufferSize,
                            						outputBuffersArray_ptr[j].bufferSize);
                            						
                    /*Print the outputted and then the expected result after mem compare */
    				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
    										   "\nData outputted by engine [output buffer]-- \n",
    										   (DxUint8_t*)outputBuffersArray_ptr[j].buffer_ptr, 
    										   outputBuffersArray_ptr[j].bufferSize); 

    		 		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
    										  "\nData Expected by engine --\n",
    										   (DxUint8_t*)pExpectedOut + 	(j+1)*outputBuffersArray_ptr[j].bufferSize,
    										   outputBuffersArray_ptr[j].bufferSize);                 	                                       						
				}
				
				TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
			
    			#if CRYS_ATP_TST_EXIT_ON_ERROR
    			if (TST_TestStatus != TESTPASS) 
    	    		return TESTFAIL;
    			#endif  
			}
        
		} /* End of TST_ATP_EncryptDecryptFlag*/ 
	                            
	}/* End of TST_index */     
	

  	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AESDYNAMIC_AddBuffers_Acceptance*/ 

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  TST_index          - test index
*  TST_MasterTestName - test name
*  flowType           - flow type (static or dynamic)
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES FLOW API's with additional engines (one test).

* Algorithm:     
*  1. Init AES flow with type and other parameters
*  2. Send several interrupts that input and output buffers are ready
*  3. Execute some HASH operation and compare the result of the HASH to expected
*  4. Send several interrupts that input and output buffers are ready, 
*  5. Execute some DES operation and compare the result of the DES to expected
*  6. Execute some AES operation and compare the result of the AES to expected
*  7. Finish AES flow 
*  8. Execute some AES operation and compare with expected result
*  9. Get result
*  10.Terminate flow
*  11.Read data from the output buffers and compare to expected results
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance_WithAdditionalEngines ( 
                                                                int              TST_index, 
                                                                char             *TST_MasterTestName,
                                                                CRYS_FLOW_Type_t flowType ) 
{

	/********* TST Variables Definitions ************/
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	DxUint8_t                   *pExpectedOut;
	/* flow id */
	DxUint32_t                   flowId = 0;
	int                          i = 0;
	int                          j = 0; 
    DxUint16_t                   blockSize;
	
	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
																											TST_OperationMode = 0 == Encrypt
																											TST_OperationMode = 1 == Decrypt*/
	{
	    /* input block size */
		blockSize       = TST_FLOW_InputDataSize/TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		
		/* prepare input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
		{
			/* in modes below there is only encrypt option */
			if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CTR_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
				break;

			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
		}
		
		
		/* Initialize AES flow parameters and init AES flow by calling function */
		/* CRYS_FLOW_AESInit                                                    */
		/************************************************************************/
		TST_TestStatus = CRYS_ATP_TST_InitAESFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo, (CRYS_AES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
		
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
        #endif  

	    /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/
	    
	    /* Send interrupt that input buffers are ready */
	    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);

		/* Send interrupt that output buffers are ready */
		if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
		    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
		
		
		/* execute HASH operation and compare with expected result */
		/***********************************************************/
	    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckHASHOperation();
	    
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"HASH operation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

        /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/
	    
        /* Send interrupt that input buffers are ready */
	    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);

		/* Send interrupt that output buffers are ready */
		if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
		    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
		
		
		/* execute DES operation and compare with expected result */
		/***********************************************************/
		TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckDESOperation();
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"DES operation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  

        /* execute some AES operation and compare with expected result*/
        /**************************************************************/
       TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

        /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/
	    
        /* Send interrupt that input buffers are ready */
        if ((TST_FLOW_IsFinishSizeZero == DX_TRUE) ||
            (TST_FLOWParamsInfo.inputBuffersArraySize < 3))
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 2; ++i)
    	    {
    	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
    	    }
    	    
    	    blockSize = 0;
        }
        else
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 3; ++i)
    	    {
    	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
    	    }
        }
	    
	    /* Send interrupt that output buffers are ready */
	    if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
     	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
	    {
    	    for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize - 2; ++i)
    	    {
    	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
    	    }
	    }

		/* Finish AES flow  */
		/********************/
		TST_ATP_Error = CRYS_FLOW_Finish(flowId,
			                             blockSize); 
			                              
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  	                             
        				                             
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             

		/* execute some AES operation and compare with expected result*/
		/**************************************************************/
		TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

		
		/* Get result */
		/******************************************************/
		do
        {

            TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
			                                    (DxUint8_t*)TST_FLOW_AESMAC_OutputData,
			                                     CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);
                                                                                                                          

        }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
		
		
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
		
		/* terminate flow */
		/******************/
		TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	
	    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  

		/* Read data from the output buffers and compare to expected result */
		/********************************************************************/
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			pExpectedOut = TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
		else
			pExpectedOut = (DxUint8_t*)TST_Input_Data;

		if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
			(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
			(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
		{
			/* compare and prints results of test */
			TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
				(DxUint8_t*)TST_FLOW_AESMAC_OutputData,
				pExpectedOut,
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);

			/*Print the outputted and then the expected result after mem compare */
			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData outputted by engine [output buffer]-- \n",
				(DxUint8_t*)TST_FLOW_AESMAC_OutputData, 
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4); 

			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData Expected by engine --\n",
				(DxUint8_t*)pExpectedOut,
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);                 	           
		}
		else
		{
			for (j = 0; j < TST_FLOWParamsInfo.outputBuffersArraySize; ++j)
			{
				/* compare and prints results of test */
				TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
					TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr,
					pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);


				/*Print the outputted and then the expected result after mem compare */
				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData outputted by engine [output buffer]-- \n",
					(DxUint8_t*)TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr, 
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize); 

				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData Expected by engine --\n",
					(DxUint8_t*)pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);                 	                                     			
			}
		}

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	} /* End of TST_ATP_EncryptDecryptFlag*/ 

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for static AES FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional enginesfor each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines"; 
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	/***********************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance_WithAdditionalEngines(TST_index,
		                                                               TST_MasterTestName,
		                                                               CRYS_FLOW_TypeStatic);
        
                                                                       
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  		                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AESSTATIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for dynamic AES FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional enginesfor each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines"; 
	

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_AES_ONETEST_Acceptance_WithAdditionalEngines(TST_index,
		                                                               TST_MasterTestName,
		                                                               CRYS_FLOW_TypeDynamic);
		                                                               
        
		                                                               		                                                                      
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  		                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_AESDYNAMIC_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AESSTATIC_ONETEST_Cyclic_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic AES FLOW API's
 
 * Algorithm:   
 * 1. Initialize AES flow with static type and other parameters 
 * 2. Init AES flow
 * 3. Send interrupts to execute cyclic flow operation 
 * 4. Finish AES flow
 * 5. Get results
 * 6. Terminate flow
 * 7. Read data from the output buffers and compare to expected result
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESSTATIC_ONETEST_Cyclic_Acceptance (int     TST_index, 
                                                                   char   *TST_MasterTestName) 
{

	/********* TST Variables Definitions ************/
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	DxUint8_t                   *pExpectedOut;
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   blockSize = 0;
	DxUint32_t                   j = 0; 
	DxUint32_t                   i = 0; 
	
	volatile DxUint32_t          delay = 0;
	
	/* output to accumulate result from cyclic operation */
	DxUint32_t	                 TST_FLOW_Cyclic_OutputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/4];
	DxUint8_t                    *pAccumulatorOutputData = (DxUint8_t*)TST_FLOW_Cyclic_OutputData;
    

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	TST_FLOWParamsInfo.inputBigEndianOrderFlag  = TST_AESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_FLOWParamsInfo.inputBuffersArraySize    = TST_AESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_FLOWParamsInfo.inputInterruptId         = TST_AESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_FLOWParamsInfo.outputBigEndianOrderFlag = TST_AESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_FLOWParamsInfo.outputBuffersArraySize   = TST_AESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_FLOWParamsInfo.outputInterruptId        = TST_AESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_FLOWParamsInfo.wordsBuffersOrderFlag    = TST_AESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 1 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                                                                              TST_OperationMode = 0 == Encrypt
                                                                                                              TST_OperationMode = 1 == Decrypt*/
    {
        /* pointer to cyclic output data for accumulating */
        pAccumulatorOutputData = (DxUint8_t*)TST_FLOW_Cyclic_OutputData;
        
        /* prepare input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
		{
		    
			/* in modes below there is only encrypt option */
			if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CTR_mode)      ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
				(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
				break;

			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_cyclicOperation_data, TST_FLOW_InputDataSize/2);
		}
		
		/* Initialize AES flow data */
     	/****************************/
     	
     	/* initialize input and output buffers */
     	blockSize = TST_FLOW_DATA_VEC_SIZE_IN_BYTS/4;
     	
     	for (j = 0; j < 2; ++j )
		{
			/* point to the second block of the input buffer */
			TST_InputBuffers[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j)*blockSize;
                
			TST_InputBuffers[j].bufferSize = blockSize;                
		}
        
       
        for (j = 0; j < 2; ++j )
		{
			/* point to the second block of the output data */
			TST_OutputBuffers[CRYS_FLOW_AES][j].buffer_ptr = (DxUint8_t*)TST_FLOW_OutputData[CRYS_FLOW_AES] + (j)*blockSize;
        
			TST_OutputBuffers[CRYS_FLOW_AES][j].bufferSize = blockSize;                
		}

        
        TST_FLOWParamsInfo.inputBuffersArrays    = TST_InputBuffers;
	    TST_FLOWParamsInfo.inputBuffersArraySize = 2;
	    
	    TST_FLOWParamsInfo.outputBuffersArrays   = TST_OutputBuffers[CRYS_FLOW_AES];
	    TST_FLOWParamsInfo.outputBuffersArraySize = 2;

     	   	
     	/* initialize AES engine data */
     	CRYS_ATP_TST_FLOW_InitAESEngineData(TST_index);
     	
     	if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
     	    (TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
     	    (TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
     	
     	{
     	    TST_FLOWParamsInfo.outputBuffersArrays    = DX_NULL;
     	    TST_FLOWParamsInfo.outputBuffersArraySize = 0;
     	}
     	
     	TST_FLOWAESEngineData.EncryptDecryptFlag = (CRYS_AES_EncryptMode_t)TST_ATP_EncryptDecryptFlag;
     	
	    /* Init AES flow */
     	/*****************/
     	TST_ATP_Error = CRYS_FLOW_AESInit(CRYS_FLOW_TypeStatic,
                     	                  &TST_FLOWParamsInfo,
                                          &TST_FLOWAESEngineData,  
                                          &flowId);
                          
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_InitAES is done\n"));
                            
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_InitAES",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Send interrupts to execute cyclic flow operation */
        /*************************************************************************/
        
        /* several times send input and output interupts to execute */
        /* several time flow operation */
        for (j = 0; j < 2; j++)
        {
            /* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
            
            /* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
            
            if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
         	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
         	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
			{
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);
                
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);
			    
			}
      
            /* wating for DMA operation to be finished and correct
               output data to be updated  */                       
            for (delay = 0; delay < 100000; delay++);
            
            /* for MAC mode the result will be after CRYS_FLOW_GetResults operation */
            if ((TST_FLOWAESEngineData.OperationMode != CRYS_AES_MAC_mode)      &&
         	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_XCBC_MAC_mode) &&
         	    (TST_FLOWAESEngineData.OperationMode != CRYS_AES_CMAC_mode))
            {
                /* accumulate output data after each cycle iteration */
                for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize; ++i)
    			{
    			    DX_VOS_FastMemCpy(pAccumulatorOutputData,
                                      TST_FLOWParamsInfo.outputBuffersArrays[i].buffer_ptr,
                                      TST_FLOWParamsInfo.outputBuffersArrays[i].bufferSize);       
                                      
                    pAccumulatorOutputData += TST_FLOWParamsInfo.outputBuffersArrays[i].bufferSize;                                                                                
    			}
            }
		
        }
        
        /* Finish AES flow */
        /*******************/
        TST_ATP_Error = CRYS_FLOW_Finish(flowId,
                                         0);  
                         
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
                         
       
        /* Get results */
        /***************/
        do
        {
            
            TST_ATP_Error  = CRYS_FLOW_GetResults(flowId, 	
                                              TST_FLOW_AESMAC_OutputData,
									          sizeof(CRYS_AES_MAX_MAC_RESULT_t));                                                                                                                      
        }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);

        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Terminate flow */
        /******************/
        TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Read data from the output buffers and compare to expected result */
        /********************************************************************/
                         
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			pExpectedOut = TST_AESFLOW_Acceptance_TestData[TST_index].TST_Expected_cyclicOperation_data;
		else
			pExpectedOut = (DxUint8_t*)TST_Input_Data; 

        if ((TST_FLOWAESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
			(TST_FLOWAESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
			(TST_FLOWAESEngineData.OperationMode == CRYS_AES_CMAC_mode))
		{
	        /* compare and prints results of test */
			TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
				(DxUint8_t*)TST_FLOW_AESMAC_OutputData,
				pExpectedOut,
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);

			/*Print the outputted and then the expected result after mem compare */
			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData outputted by engine [output buffer]-- \n",
				(DxUint8_t*)TST_FLOW_AESMAC_OutputData, 
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4); 

			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData Expected by engine --\n",
				(DxUint8_t*)pExpectedOut,
				CRYS_AES_IV_COUNTER_SIZE_IN_WORDS*4);                 	           
		}
		else
		{
			
			/* compare and prints results of test */
			TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
			            TST_FLOW_Cyclic_OutputData,
			            pExpectedOut,
			            TST_FLOW_DATA_VEC_SIZE_IN_BYTS);
			            
		    /*Print the outputted and then the expected result after mem compare */
			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData outputted by engine [output buffer]-- \n",
				(DxUint8_t*)TST_FLOW_Cyclic_OutputData, 
				TST_FLOW_DATA_VEC_SIZE_IN_BYTS); 

			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData Expected by engine --\n",
				(DxUint8_t*)pExpectedOut,
				TST_FLOW_DATA_VEC_SIZE_IN_BYTS);                 	                                     			                       							
		}

        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_AESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
        
        
	
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
	    	return TESTFAIL;
	    #endif  
    
	} /* End of TST_ATP_EncryptDecryptFlag*/ 
		
		
	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_AESSTATIC_ONETEST_Cyclic_Acceptance*/ 



/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic AES FLOW API's
 
 * Algorithm:     
 *	This function goes throw all test vectors and check cyclic acceptance.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance"; 
	
  
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* This function goes throw all test vectors and check cyclic acceptance */
	/*************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_FLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	TST_TestStatus = CRYS_ATP_TST_FLOW_AESSTATIC_ONETEST_Cyclic_Acceptance(TST_index, TST_MasterTestName);
     	
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  		                                                               
                        
	}/* End of TST_index */     
    
  	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_AESSTATIC_Cyclic_Acceptance*/ 

#endif
